<?php
/**
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 * http://www.gnu.org/copyleft/gpl.html
 *
 * @file
 */
namespace MediaWiki\Minerva\Menu\User;

use MediaWiki\Html\TemplateParser;
use MessageLocalizer;

/**
 * Director responsible for building the user menu.
 */
final class UserMenuDirector {
	private IUserMenuBuilder $builder;
	private MessageLocalizer $localizer;

	public function __construct( IUserMenuBuilder $builder, MessageLocalizer $localizer ) {
		$this->builder = $builder;
		$this->localizer = $localizer;
	}

	/**
	 * Build the menu data array that can be passed to views/javascript
	 * @param array $personalTools Personal tools list generated by Skin::buildPersonalUrls
	 * @return string|null
	 */
	public function renderMenuData( array $personalTools ): ?string {
		$group = $this->builder->getGroup( $personalTools );
		$entries = $group->getEntries();

		$templateParser = new TemplateParser( __DIR__ . '/../../Skins' );
		$toggleID = 'minerva-user-menu-toggle';
		$checkboxID = 'minerva-user-menu-checkbox';
		return !$entries
			? null
			: $templateParser->processTemplate( 'ToggleList', [
				'class' => 'minerva-user-menu',
				'checkboxID' => $checkboxID,
				'toggleID' => $toggleID,
				'event' => 'ui.usermenu',
				'data-btn' => [
					'tag-name' => 'label',
					'data-icon' => [
						'icon' => 'userAvatarOutline',
					],
					'classes' => 'toggle-list__toggle',
					'array-attributes' => [
						[
							'key' => 'id',
							'value' => $toggleID,
						],
						[
							'key' => 'for',
							'value' => $checkboxID,
						],
						[
							'key' => 'aria-hidden',
							'value' => 'true'
						],
					],
					'label' => $this->localizer->msg( 'minerva-user-menu-button' )->escaped(),
				],
				'listID' => $group->getId(),
				// See ToggleList/*.less.
				'listClass' => 'minerva-user-menu-list toggle-list__list--drop-down',
				'items' => $entries
			] );
	}
}
