﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/s3/S3_EXPORTS.h>
#include <aws/s3/model/MetricsAndOperator.h>
#include <aws/s3/model/Tag.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Xml {
class XmlNode;
}  // namespace Xml
}  // namespace Utils
namespace S3 {
namespace Model {

/**
 * <p>Specifies a metrics configuration filter. The metrics configuration only
 * includes objects that meet the filter's criteria. A filter must be a prefix, an
 * object tag, an access point ARN, or a conjunction (MetricsAndOperator). For more
 * information, see <a
 * href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketMetricsConfiguration.html">PutBucketMetricsConfiguration</a>.</p><p><h3>See
 * Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/s3-2006-03-01/MetricsFilter">AWS
 * API Reference</a></p>
 */
class MetricsFilter {
 public:
  AWS_S3_API MetricsFilter() = default;
  AWS_S3_API MetricsFilter(const Aws::Utils::Xml::XmlNode& xmlNode);
  AWS_S3_API MetricsFilter& operator=(const Aws::Utils::Xml::XmlNode& xmlNode);

  AWS_S3_API void AddToNode(Aws::Utils::Xml::XmlNode& parentNode) const;

  ///@{
  /**
   * <p>The prefix used when evaluating a metrics filter.</p>
   */
  inline const Aws::String& GetPrefix() const { return m_prefix; }
  inline bool PrefixHasBeenSet() const { return m_prefixHasBeenSet; }
  template <typename PrefixT = Aws::String>
  void SetPrefix(PrefixT&& value) {
    m_prefixHasBeenSet = true;
    m_prefix = std::forward<PrefixT>(value);
  }
  template <typename PrefixT = Aws::String>
  MetricsFilter& WithPrefix(PrefixT&& value) {
    SetPrefix(std::forward<PrefixT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The tag used when evaluating a metrics filter.</p>
   */
  inline const Tag& GetTag() const { return m_tag; }
  inline bool TagHasBeenSet() const { return m_tagHasBeenSet; }
  template <typename TagT = Tag>
  void SetTag(TagT&& value) {
    m_tagHasBeenSet = true;
    m_tag = std::forward<TagT>(value);
  }
  template <typename TagT = Tag>
  MetricsFilter& WithTag(TagT&& value) {
    SetTag(std::forward<TagT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The access point ARN used when evaluating a metrics filter.</p>
   */
  inline const Aws::String& GetAccessPointArn() const { return m_accessPointArn; }
  inline bool AccessPointArnHasBeenSet() const { return m_accessPointArnHasBeenSet; }
  template <typename AccessPointArnT = Aws::String>
  void SetAccessPointArn(AccessPointArnT&& value) {
    m_accessPointArnHasBeenSet = true;
    m_accessPointArn = std::forward<AccessPointArnT>(value);
  }
  template <typename AccessPointArnT = Aws::String>
  MetricsFilter& WithAccessPointArn(AccessPointArnT&& value) {
    SetAccessPointArn(std::forward<AccessPointArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A conjunction (logical AND) of predicates, which is used in evaluating a
   * metrics filter. The operator must have at least two predicates, and an object
   * must match all of the predicates in order for the filter to apply.</p>
   */
  inline const MetricsAndOperator& GetAnd() const { return m_and; }
  inline bool AndHasBeenSet() const { return m_andHasBeenSet; }
  template <typename AndT = MetricsAndOperator>
  void SetAnd(AndT&& value) {
    m_andHasBeenSet = true;
    m_and = std::forward<AndT>(value);
  }
  template <typename AndT = MetricsAndOperator>
  MetricsFilter& WithAnd(AndT&& value) {
    SetAnd(std::forward<AndT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_prefix;
  bool m_prefixHasBeenSet = false;

  Tag m_tag;
  bool m_tagHasBeenSet = false;

  Aws::String m_accessPointArn;
  bool m_accessPointArnHasBeenSet = false;

  MetricsAndOperator m_and;
  bool m_andHasBeenSet = false;
};

}  // namespace Model
}  // namespace S3
}  // namespace Aws
