﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/iotsecuretunneling/IoTSecureTunnelingRequest.h>
#include <aws/iotsecuretunneling/IoTSecureTunneling_EXPORTS.h>
#include <aws/iotsecuretunneling/model/ClientMode.h>
#include <aws/iotsecuretunneling/model/DestinationConfig.h>

#include <utility>

namespace Aws {
namespace IoTSecureTunneling {
namespace Model {

/**
 */
class RotateTunnelAccessTokenRequest : public IoTSecureTunnelingRequest {
 public:
  AWS_IOTSECURETUNNELING_API RotateTunnelAccessTokenRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "RotateTunnelAccessToken"; }

  AWS_IOTSECURETUNNELING_API Aws::String SerializePayload() const override;

  AWS_IOTSECURETUNNELING_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The tunnel for which you want to rotate the access tokens.</p>
   */
  inline const Aws::String& GetTunnelId() const { return m_tunnelId; }
  inline bool TunnelIdHasBeenSet() const { return m_tunnelIdHasBeenSet; }
  template <typename TunnelIdT = Aws::String>
  void SetTunnelId(TunnelIdT&& value) {
    m_tunnelIdHasBeenSet = true;
    m_tunnelId = std::forward<TunnelIdT>(value);
  }
  template <typename TunnelIdT = Aws::String>
  RotateTunnelAccessTokenRequest& WithTunnelId(TunnelIdT&& value) {
    SetTunnelId(std::forward<TunnelIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The mode of the client that will use the client token, which can be either
   * the source or destination, or both source and destination.</p>
   */
  inline ClientMode GetClientMode() const { return m_clientMode; }
  inline bool ClientModeHasBeenSet() const { return m_clientModeHasBeenSet; }
  inline void SetClientMode(ClientMode value) {
    m_clientModeHasBeenSet = true;
    m_clientMode = value;
  }
  inline RotateTunnelAccessTokenRequest& WithClientMode(ClientMode value) {
    SetClientMode(value);
    return *this;
  }
  ///@}

  ///@{

  inline const DestinationConfig& GetDestinationConfig() const { return m_destinationConfig; }
  inline bool DestinationConfigHasBeenSet() const { return m_destinationConfigHasBeenSet; }
  template <typename DestinationConfigT = DestinationConfig>
  void SetDestinationConfig(DestinationConfigT&& value) {
    m_destinationConfigHasBeenSet = true;
    m_destinationConfig = std::forward<DestinationConfigT>(value);
  }
  template <typename DestinationConfigT = DestinationConfig>
  RotateTunnelAccessTokenRequest& WithDestinationConfig(DestinationConfigT&& value) {
    SetDestinationConfig(std::forward<DestinationConfigT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_tunnelId;
  bool m_tunnelIdHasBeenSet = false;

  ClientMode m_clientMode{ClientMode::NOT_SET};
  bool m_clientModeHasBeenSet = false;

  DestinationConfig m_destinationConfig;
  bool m_destinationConfigHasBeenSet = false;
};

}  // namespace Model
}  // namespace IoTSecureTunneling
}  // namespace Aws
