// Copyright © 2023-2025 Advanced Micro Devices, Inc.
// SPDX-License-Identifier: MIT

#include "../shim.bwd_kernel_fuse.h"
#include <aotriton/_internal/triton_kernel.h>
#include <aotriton/_internal/kernel_cluster.h>
#include <aotriton/cpp_tune.h>
#include <string_view>
#ifndef NDEBUG
#include <iostream>
#endif

#define CURRENT_ENTRY_PUBLIC Autotune_bwd_kernel_fuse__A4__F352

#define ARRAY_SIZE(array)  (sizeof(array) / sizeof(array[0]))

namespace { // Anonymous namespace

using namespace std::literals::string_view_literals;

#if AOTRITON_BUILD_FOR_TUNING
// PSels and Copts in JSON String
static const char* kernel_psels[] = {
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 32})xyzw"
};
static const char* kernel_copts[] = {
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 4, "num_stages": 1})xyzw"
};
#endif

struct PerfFields {
  int16_t BLOCK_M;
    int16_t BLOCK_N;
};

static PerfFields image_perf_list [] = {
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 32 },
    { .BLOCK_M = 32, .BLOCK_N = 32 }
};

// u8R generates char8_t which is poorly supported almost everywhere.
constexpr pstring_view PACKAGE_PATH
#if defined(_WIN32)
{ LR"xyzw(amd-gfx11xx/flash/bwd_kernel_fuse/FONLY__＊fp32@16_16_0_F_F_0___gfx11xx)xyzw" };
#else
{ R"xyzw(amd-gfx11xx/flash/bwd_kernel_fuse/FONLY__＊fp32@16_16_0_F_F_0___gfx11xx)xyzw" };
#endif
constexpr std::string_view FUNC_NAME { R"xyzw(＊fp32@16_16_0_F_F_0)xyzw" };
constexpr std::string_view ARCH_NAME { R"xyzw(gfx1103)xyzw" };

// Checksum can be confirmed with `echo -n '<string>' | b2sum -l 64`
// For example:
//   $ echo -n 'amd-gfx110x/flash/attn_fwd/FONLY__^bf16@16,128,False,False,0,0___gfx1100__P__32_32_0_2_False__CO__wave3_warp2_stg1-Gpu-gfx1100' | b2sum -l 64
//   c4b51ee645d79580  -
static AOTRITON_NS::TritonKernelCompactMeta meta_list[] = {
    { 0x06b67d19u, 0x076575afu, 0, 109 }, // 06b67d19076575af = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_fuse/FONLY__＊fp32@16_16_0_F_F_0___gfx11xx__P__16_16__CO__wave1_warp2_stg1--Arch_gfx1103
    { 0x214c8d04u, 0xd2897c7eu, 0, 126 }, // 214c8d04d2897c7e = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_fuse/FONLY__＊fp32@16_16_0_F_F_0___gfx11xx__P__16_16__CO__wave1_warp4_stg1--Arch_gfx1103
    { 0x75b90aa2u, 0x2db14d6eu, 0, 69 }, // 75b90aa22db14d6e = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_fuse/FONLY__＊fp32@16_16_0_F_F_0___gfx11xx__P__16_16__CO__wave2_warp4_stg1--Arch_gfx1103
    { 0x3c7a2247u, 0xeaea8ddau, 0, 29 }, // 3c7a2247eaea8dda = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_fuse/FONLY__＊fp32@16_16_0_F_F_0___gfx11xx__P__16_16__CO__wave3_warp4_stg1--Arch_gfx1103
    { 0x60d48d3bu, 0xc828c11fu, 0, 46 }, // 60d48d3bc828c11f = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_fuse/FONLY__＊fp32@16_16_0_F_F_0___gfx11xx__P__16_16__CO__wave4_warp4_stg1--Arch_gfx1103
    { 0x0938aee5u, 0x436f25e8u, 23, 109 }, // 0938aee5436f25e8 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_fuse/FONLY__＊fp32@16_16_0_F_F_0___gfx11xx__P__32_16__CO__wave1_warp2_stg1--Arch_gfx1103
    { 0x5dc93300u, 0x47cd776bu, 23, 6 }, // 5dc9330047cd776b = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_fuse/FONLY__＊fp32@16_16_0_F_F_0___gfx11xx__P__32_16__CO__wave2_warp2_stg1--Arch_gfx1103
    { 0x6ff38053u, 0xa653272eu, 23, 143 }, // 6ff38053a653272e = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_fuse/FONLY__＊fp32@16_16_0_F_F_0___gfx11xx__P__32_16__CO__wave3_warp2_stg1--Arch_gfx1103
    { 0xc95ad6e2u, 0x8b38cd13u, 23, 86 }, // c95ad6e28b38cd13 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_fuse/FONLY__＊fp32@16_16_0_F_F_0___gfx11xx__P__32_16__CO__wave4_warp2_stg1--Arch_gfx1103
    { 0x105e150bu, 0x8ded8b53u, 63, 69 }, // 105e150b8ded8b53 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_fuse/FONLY__＊fp32@16_16_0_F_F_0___gfx11xx__P__32_32__CO__wave2_warp4_stg1--Arch_gfx1103
    { 0x07feb4d4u, 0x972e9df8u, 63, 46 }, // 07feb4d4972e9df8 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_fuse/FONLY__＊fp32@16_16_0_F_F_0___gfx11xx__P__32_32__CO__wave4_warp4_stg1--Arch_gfx1103
};

static constexpr int kTotalNumKernels = ARRAY_SIZE(meta_list);

using AOTRITON_NS::v3::flash::autotune::bwd_kernel_fuse_packed_string;

static AOTRITON_NS::TritonKernelCluster<kTotalNumKernels>
kernel_cluster(meta_list,
               bwd_kernel_fuse_packed_string);

static int8_t lut[1][7][7] =
{
// GPU gfx1103_mod0
{{ 1, 1, 0, 1, 1, 1, 1},
 { 3, 1, 4, 4, 1, 1, 1},
 { 3, 3, 4, 1, 1, 1, 1},
 { 4, 2, 6, 8, 1, 1, 1},
 { 2, 3, 6, 7, 5, 5, 5},
 { 3, 1, 5, 8, 5, 5, 9},
 { 1, 1, 1, 7, 7, 5,10}}
// End of GPU gfx1103_mod0
}
;

}; // End of anonymous namespace

namespace AOTRITON_NS::v3::flash::autotune {

// using AOTRITON_NS::v2::flash::BwdKernelFuseContext;

void CURRENT_ENTRY_PUBLIC(BwdKernelFuseContext& context, int mod_number) {
#if AOTRITON_BUILD_FOR_TUNING
    int preferred_index = context._has_preferred_kernel;
    context._total_number_of_kernels = kTotalNumKernels;
#ifndef NDEBUG
    std::cerr << "Autotune_bwd_kernel_fuse__A4__F352 "
              << "kTotalNumKernels = " << kTotalNumKernels << " "
              << "_has_preferred_kernel = " << preferred_index << " "
              << std::endl;
#endif
    if (preferred_index != -1) {
        if (preferred_index >= kTotalNumKernels)
            return ;
        context.kernel_on_device = kernel_cluster.get(preferred_index);
        context.pp_args_index = 0;
        context.package_path = PACKAGE_PATH;
        context.func_name = FUNC_NAME;
        context.arch_name = ARCH_NAME;
        context._preferred_kernel_psels = kernel_psels[preferred_index];
        context._preferred_kernel_copts = kernel_copts[preferred_index];
        const auto& perf = image_perf_list[preferred_index];
        context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
        return ;
    }
#endif
    auto kernel_index = bwd_kernel_fuse__lut_lambda__0(*context.params, mod_number, lut);
    if (kernel_index < 0) {
      return ;
    }
    context.kernel_on_device = kernel_cluster.get(kernel_index);
    context.pp_args_index = 0;
    context.package_path = PACKAGE_PATH;
    context.func_name = FUNC_NAME;
    context.arch_name = ARCH_NAME;
#ifndef NDEBUG
    std::cerr << __FILE__ << " kernel_index = " << int(kernel_index) << std::endl;
#endif
    const auto& perf = image_perf_list[kernel_index];
    context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
}

#undef CURRENT_ENTRY_PUBLIC
}

// Human-readable Signature 
// Q = "*fp32:16"
// sm_scale = "fp32"
// L = "*fp32:16"
// stride_bz = 0
// stride_dbz = 0
// num_head_q = "i32"
// cu_seqlens_q = "*i32:16"
// num_seqlens = "i32"
// dropout_p = 0
// philox_seed_ptr = 0
// philox_offset1 = 0
// philox_offset2 = 0
// Window_left = 0
// BLOCK_DMODEL = 16
// CAUSAL_TYPE = 0
// ENABLE_DROPOUT = False
// PADDED_HEAD = False
// BIAS_TYPE = 0

// SELECT * FROM FLASH$bwd_kernel_fuse WHERE gpu IN ('gfx1100_mod0') AND inputs$Q_dtype = 'torch.float32' AND inputs$BLOCK_DMODEL = 16 AND inputs$CAUSAL_TYPE = 0 AND inputs$ENABLE_DROPOUT = False AND inputs$PADDED_HEAD = False AND inputs$BIAS_TYPE = 0

// vim: set fileencoding=utf-8

