use gdk4::Texture;
use glib::{Properties, clone, subclass};
use gtk4::{Accessible, Box, Buildable, CompositeTemplate, ConstraintTarget, Widget, prelude::*, subclass::prelude::*};
use std::cell::{Cell, RefCell};

use crate::app::App;
use crate::gobject_models::{GFeedListItem, GFeedListItemID};
use crate::infrastructure::{FaviconCache, TokioRuntime};
use crate::sidebar::FeedListItemID;

mod imp {
    use super::*;

    #[derive(Debug, CompositeTemplate, Properties)]
    #[properties(wrapper_type = super::ItemRowDnD)]
    #[template(file = "data/resources/ui_templates/sidebar/feedlist_dnd_icon.blp")]
    #[derive(Default)]
    pub struct ItemRowDnD {
        #[property(get, set)]
        pub id: RefCell<GFeedListItemID>,

        #[property(get, set)]
        pub label: RefCell<String>,

        #[property(get, set, name = "item-count")]
        pub item_count: Cell<u32>,

        #[property(get, set, name = "favicon-texture", nullable)]
        pub favicon_texture: RefCell<Option<Texture>>,
    }

    #[glib::object_subclass]
    impl ObjectSubclass for ItemRowDnD {
        const NAME: &'static str = "ItemRowDnD";
        type ParentType = Box;
        type Type = super::ItemRowDnD;

        fn class_init(klass: &mut Self::Class) {
            klass.bind_template();
            klass.bind_template_callbacks();
        }

        fn instance_init(obj: &subclass::InitializingObject<Self>) {
            obj.init_template();
        }
    }

    #[glib::derived_properties]
    impl ObjectImpl for ItemRowDnD {}

    impl WidgetImpl for ItemRowDnD {}

    impl BoxImpl for ItemRowDnD {}

    #[gtk4::template_callbacks]
    impl ItemRowDnD {
        #[template_callback]
        pub fn is_count_visible(&self, count: u32) -> bool {
            count > 0
        }

        #[template_callback]
        pub fn is_favicon_visible(&self, texture: Option<Texture>) -> bool {
            texture.is_some() && self.id.borrow().is_feed()
        }

        #[template_callback]
        pub fn is_generic_favicon_visible(&self, texture: Option<Texture>) -> bool {
            texture.is_none() && self.id.borrow().is_feed()
        }
    }
}

glib::wrapper! {
    pub struct ItemRowDnD(ObjectSubclass<imp::ItemRowDnD>)
        @extends Widget, Box,
        @implements Accessible, Buildable, ConstraintTarget;
}

impl Default for ItemRowDnD {
    fn default() -> Self {
        glib::Object::new::<Self>()
    }
}

impl ItemRowDnD {
    pub fn new(model: GFeedListItem) -> Self {
        let widget = Self::default();

        widget.set_id(model.item_id());
        widget.set_label(model.label());
        widget.set_item_count(model.item_count());
        widget.load_favicon();

        widget
    }

    fn load_favicon(&self) {
        let FeedListItemID::Feed(feed_mapping) = self.id().into() else {
            return;
        };

        let header_map = crate::app::App::default()
            .settings()
            .get_feed_header_map(&feed_mapping.feed_id);

        TokioRuntime::execute_with_callback(
            move || async move {
                FaviconCache::get_icon(&feed_mapping.feed_id, App::news_flash(), App::client(), header_map).await
            },
            clone!(
                #[weak(rename_to = row)]
                self,
                move |texture| row.set_favicon_texture(texture)
            ),
        );
    }
}
