{-# LANGUAGE TypeApplications #-}


-- | Copyright  : Will Thompson and Iñaki García Etxebarria
-- License    : LGPL-2.1
-- Maintainer : Iñaki García Etxebarria
-- 
-- The basic block in a scene graph to be rendered using t'GI.Gsk.Objects.Renderer.Renderer'.
-- 
-- Each node has a parent, except the top-level node; each node may have
-- children nodes.
-- 
-- Each node has an associated drawing surface, which has the size of
-- the rectangle set when creating it.
-- 
-- Render nodes are meant to be transient; once they have been associated
-- to a t'GI.Gsk.Objects.Renderer.Renderer' it\'s safe to release any reference you have on
-- them. All t'GI.Gsk.Objects.RenderNode.RenderNode's are immutable, you can only specify their
-- properties during construction.

#if (MIN_VERSION_haskell_gi_overloading(1,0,0) && !defined(__HADDOCK_VERSION__))
#define ENABLE_OVERLOADING
#endif

module GI.Gsk.Objects.RenderNode
    ( 

-- * Exported types
    RenderNode(..)                          ,
    IsRenderNode                            ,
    toRenderNode                            ,


 -- * Methods
-- | 
-- 
--  === __Click to display all available methods, including inherited ones__
-- ==== Methods
-- [draw]("GI.Gsk.Objects.RenderNode#g:method:draw"), [ref]("GI.Gsk.Objects.RenderNode#g:method:ref"), [serialize]("GI.Gsk.Objects.RenderNode#g:method:serialize"), [unref]("GI.Gsk.Objects.RenderNode#g:method:unref"), [writeToFile]("GI.Gsk.Objects.RenderNode#g:method:writeToFile").
-- 
-- ==== Getters
-- [getBounds]("GI.Gsk.Objects.RenderNode#g:method:getBounds"), [getNodeType]("GI.Gsk.Objects.RenderNode#g:method:getNodeType"), [getOpaqueRect]("GI.Gsk.Objects.RenderNode#g:method:getOpaqueRect").
-- 
-- ==== Setters
-- /None/.

#if defined(ENABLE_OVERLOADING)
    ResolveRenderNodeMethod                 ,
#endif

-- ** deserialize #method:deserialize#

    renderNodeDeserialize                   ,


-- ** draw #method:draw#

#if defined(ENABLE_OVERLOADING)
    RenderNodeDrawMethodInfo                ,
#endif
    renderNodeDraw                          ,


-- ** getBounds #method:getBounds#

#if defined(ENABLE_OVERLOADING)
    RenderNodeGetBoundsMethodInfo           ,
#endif
    renderNodeGetBounds                     ,


-- ** getNodeType #method:getNodeType#

#if defined(ENABLE_OVERLOADING)
    RenderNodeGetNodeTypeMethodInfo         ,
#endif
    renderNodeGetNodeType                   ,


-- ** getOpaqueRect #method:getOpaqueRect#

#if defined(ENABLE_OVERLOADING)
    RenderNodeGetOpaqueRectMethodInfo       ,
#endif
    renderNodeGetOpaqueRect                 ,


-- ** ref #method:ref#

#if defined(ENABLE_OVERLOADING)
    RenderNodeRefMethodInfo                 ,
#endif
    renderNodeRef                           ,


-- ** serialize #method:serialize#

#if defined(ENABLE_OVERLOADING)
    RenderNodeSerializeMethodInfo           ,
#endif
    renderNodeSerialize                     ,


-- ** unref #method:unref#

#if defined(ENABLE_OVERLOADING)
    RenderNodeUnrefMethodInfo               ,
#endif
    renderNodeUnref                         ,


-- ** writeToFile #method:writeToFile#

#if defined(ENABLE_OVERLOADING)
    RenderNodeWriteToFileMethodInfo         ,
#endif
    renderNodeWriteToFile                   ,




    ) where

import Data.GI.Base.ShortPrelude
import qualified Data.GI.Base.ShortPrelude as SP
import qualified Data.GI.Base.Overloading as O
import qualified Prelude as P

import qualified Data.GI.Base.Attributes as GI.Attributes
import qualified Data.GI.Base.BasicTypes as B.Types
import qualified Data.GI.Base.ManagedPtr as B.ManagedPtr
import qualified Data.GI.Base.GArray as B.GArray
import qualified Data.GI.Base.GClosure as B.GClosure
import qualified Data.GI.Base.GError as B.GError
import qualified Data.GI.Base.GHashTable as B.GHT
import qualified Data.GI.Base.GVariant as B.GVariant
import qualified Data.GI.Base.GValue as B.GValue
import qualified Data.GI.Base.GParamSpec as B.GParamSpec
import qualified Data.GI.Base.CallStack as B.CallStack
import qualified Data.GI.Base.Properties as B.Properties
import qualified Data.GI.Base.Signals as B.Signals
import qualified Control.Monad.IO.Class as MIO
import qualified Data.Coerce as Coerce
import qualified Data.Text as T
import qualified Data.Kind as DK
import qualified Data.ByteString.Char8 as B
import qualified Data.Map as Map
import qualified Foreign.Ptr as FP
import qualified GHC.OverloadedLabels as OL
import qualified GHC.Records as R
import qualified Data.Word as DW
import qualified Data.Int as DI
import qualified System.Posix.Types as SPT
import qualified Foreign.C.Types as FCT

-- Workaround for https://gitlab.haskell.org/ghc/ghc/-/issues/23392
#if MIN_VERSION_base(4,18,0)
import qualified GI.Cairo.Structs.Context as Cairo.Context
import qualified GI.GLib.Structs.Bytes as GLib.Bytes
import qualified GI.Graphene.Structs.Rect as Graphene.Rect
import qualified GI.Gsk.Callbacks as Gsk.Callbacks
import {-# SOURCE #-} qualified GI.Gsk.Enums as Gsk.Enums

#else
import qualified GI.Cairo.Structs.Context as Cairo.Context
import qualified GI.GLib.Structs.Bytes as GLib.Bytes
import qualified GI.Graphene.Structs.Rect as Graphene.Rect
import qualified GI.Gsk.Callbacks as Gsk.Callbacks
import {-# SOURCE #-} qualified GI.Gsk.Enums as Gsk.Enums

#endif

-- | Memory-managed wrapper type.
newtype RenderNode = RenderNode (SP.ManagedPtr RenderNode)
    deriving (Eq)

instance SP.ManagedPtrNewtype RenderNode where
    toManagedPtr (RenderNode p) = p

foreign import ccall "gsk_render_node_get_type"
    c_gsk_render_node_get_type :: IO B.Types.GType

instance B.Types.TypedObject RenderNode where
    glibType = c_gsk_render_node_get_type

-- | Type class for types which can be safely cast to t'RenderNode', for instance with `toRenderNode`.
class (SP.BoxedPtr o, SP.TypedObject o, O.IsDescendantOf RenderNode o) => IsRenderNode o
instance (SP.BoxedPtr o, SP.TypedObject o, O.IsDescendantOf RenderNode o) => IsRenderNode o

instance O.HasParentTypes RenderNode
type instance O.ParentTypes RenderNode = '[]

-- | Cast to t'RenderNode', for types for which this is known to be safe. For general casts, use 'Data.GI.Base.ManagedPtr.castTo'.
toRenderNode :: (MIO.MonadIO m, IsRenderNode o) => o -> m RenderNode
toRenderNode = MIO.liftIO . B.ManagedPtr.unsafeCastTo RenderNode

foreign import ccall "gsk_value_get_render_node" gv_get_gsk_value_get_render_node ::
    FP.Ptr B.GValue.GValue -> IO (FP.Ptr RenderNode)

foreign import ccall "gsk_value_set_render_node" gv_set_gsk_value_set_render_node ::
    FP.Ptr B.GValue.GValue -> FP.Ptr RenderNode -> IO ()

-- | Convert t'RenderNode' to and from t'Data.GI.Base.GValue.GValue'. See 'Data.GI.Base.GValue.toGValue' and 'Data.GI.Base.GValue.fromGValue'.
instance B.GValue.IsGValue (Maybe RenderNode) where
    gvalueGType_ = c_gsk_render_node_get_type
    gvalueSet_ gv P.Nothing = gv_set_gsk_value_set_render_node gv (FP.nullPtr :: FP.Ptr RenderNode)
    gvalueSet_ gv (P.Just obj) = B.ManagedPtr.withManagedPtr obj (gv_set_gsk_value_set_render_node gv)
    gvalueGet_ gv = do
        ptr <- gv_get_gsk_value_get_render_node gv :: IO (FP.Ptr RenderNode)
        if ptr /= FP.nullPtr
        then P.Just <$> B.ManagedPtr.newPtr RenderNode ptr
        else return P.Nothing
        
    

#if defined(ENABLE_OVERLOADING)
type family ResolveRenderNodeMethod (t :: Symbol) (o :: DK.Type) :: DK.Type where
    ResolveRenderNodeMethod "draw" o = RenderNodeDrawMethodInfo
    ResolveRenderNodeMethod "ref" o = RenderNodeRefMethodInfo
    ResolveRenderNodeMethod "serialize" o = RenderNodeSerializeMethodInfo
    ResolveRenderNodeMethod "unref" o = RenderNodeUnrefMethodInfo
    ResolveRenderNodeMethod "writeToFile" o = RenderNodeWriteToFileMethodInfo
    ResolveRenderNodeMethod "getBounds" o = RenderNodeGetBoundsMethodInfo
    ResolveRenderNodeMethod "getNodeType" o = RenderNodeGetNodeTypeMethodInfo
    ResolveRenderNodeMethod "getOpaqueRect" o = RenderNodeGetOpaqueRectMethodInfo
    ResolveRenderNodeMethod l o = O.MethodResolutionFailed l o

instance (info ~ ResolveRenderNodeMethod t RenderNode, O.OverloadedMethod info RenderNode p) => OL.IsLabel t (RenderNode -> p) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.overloadedMethod @info
#else
    fromLabel _ = O.overloadedMethod @info
#endif

#if MIN_VERSION_base(4,13,0)
instance (info ~ ResolveRenderNodeMethod t RenderNode, O.OverloadedMethod info RenderNode p, R.HasField t RenderNode p) => R.HasField t RenderNode p where
    getField = O.overloadedMethod @info

#endif

instance (info ~ ResolveRenderNodeMethod t RenderNode, O.OverloadedMethodInfo info RenderNode) => OL.IsLabel t (O.MethodProxy info RenderNode) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.MethodProxy
#else
    fromLabel _ = O.MethodProxy
#endif

#endif

foreign import ccall "gsk_render_node_ref" _RenderNode_copy_gsk_render_node_ref :: Ptr a -> IO (Ptr a)

foreign import ccall "gsk_render_node_unref" _RenderNode_free_gsk_render_node_unref :: Ptr a -> IO ()

instance BoxedPtr RenderNode where
    boxedPtrCopy = \p -> B.ManagedPtr.withManagedPtr p (_RenderNode_copy_gsk_render_node_ref >=> B.ManagedPtr.wrapPtr RenderNode)
    boxedPtrFree = \p -> B.ManagedPtr.withManagedPtr p _RenderNode_free_gsk_render_node_unref


-- method RenderNode::draw
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "node"
--           , argType =
--               TInterface Name { namespace = "Gsk" , name = "RenderNode" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a render node" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cr"
--           , argType =
--               TInterface Name { namespace = "cairo" , name = "Context" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "cairo context to draw to"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gsk_render_node_draw" gsk_render_node_draw :: 
    Ptr RenderNode ->                       -- node : TInterface (Name {namespace = "Gsk", name = "RenderNode"})
    Ptr Cairo.Context.Context ->            -- cr : TInterface (Name {namespace = "cairo", name = "Context"})
    IO ()

-- | Draws the contents of a render node on a cairo context.
-- 
-- Typically, you\'ll use this function to implement fallback rendering
-- of render nodes on an intermediate Cairo context, instead of using
-- the drawing context associated to a t'GI.Gdk.Objects.Surface.Surface'\'s rendering buffer.
-- 
-- For advanced nodes that cannot be supported using Cairo, in particular
-- for nodes doing 3D operations, this function may fail.
renderNodeDraw ::
    (B.CallStack.HasCallStack, MonadIO m, IsRenderNode a) =>
    a
    -- ^ /@node@/: a render node
    -> Cairo.Context.Context
    -- ^ /@cr@/: cairo context to draw to
    -> m ()
renderNodeDraw node cr = liftIO $ do
    node' <- unsafeManagedPtrCastPtr node
    cr' <- unsafeManagedPtrGetPtr cr
    gsk_render_node_draw node' cr'
    touchManagedPtr node
    touchManagedPtr cr
    return ()

#if defined(ENABLE_OVERLOADING)
data RenderNodeDrawMethodInfo
instance (signature ~ (Cairo.Context.Context -> m ()), MonadIO m, IsRenderNode a) => O.OverloadedMethod RenderNodeDrawMethodInfo a signature where
    overloadedMethod = renderNodeDraw

instance O.OverloadedMethodInfo RenderNodeDrawMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gsk.Objects.RenderNode.renderNodeDraw",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gsk-4.0.8/docs/GI-Gsk-Objects-RenderNode.html#v:renderNodeDraw"
        })


#endif

-- method RenderNode::get_bounds
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "node"
--           , argType =
--               TInterface Name { namespace = "Gsk" , name = "RenderNode" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a render node" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "bounds"
--           , argType =
--               TInterface Name { namespace = "Graphene" , name = "Rect" }
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "return location for the boundaries"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = True
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gsk_render_node_get_bounds" gsk_render_node_get_bounds :: 
    Ptr RenderNode ->                       -- node : TInterface (Name {namespace = "Gsk", name = "RenderNode"})
    Ptr Graphene.Rect.Rect ->               -- bounds : TInterface (Name {namespace = "Graphene", name = "Rect"})
    IO ()

-- | Retrieves the boundaries of the /@node@/.
-- 
-- The node will not draw outside of its boundaries.
renderNodeGetBounds ::
    (B.CallStack.HasCallStack, MonadIO m, IsRenderNode a) =>
    a
    -- ^ /@node@/: a render node
    -> m (Graphene.Rect.Rect)
renderNodeGetBounds node = liftIO $ do
    node' <- unsafeManagedPtrCastPtr node
    bounds <- SP.callocBoxedBytes 16 :: IO (Ptr Graphene.Rect.Rect)
    gsk_render_node_get_bounds node' bounds
    bounds' <- (wrapBoxed Graphene.Rect.Rect) bounds
    touchManagedPtr node
    return bounds'

#if defined(ENABLE_OVERLOADING)
data RenderNodeGetBoundsMethodInfo
instance (signature ~ (m (Graphene.Rect.Rect)), MonadIO m, IsRenderNode a) => O.OverloadedMethod RenderNodeGetBoundsMethodInfo a signature where
    overloadedMethod = renderNodeGetBounds

instance O.OverloadedMethodInfo RenderNodeGetBoundsMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gsk.Objects.RenderNode.renderNodeGetBounds",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gsk-4.0.8/docs/GI-Gsk-Objects-RenderNode.html#v:renderNodeGetBounds"
        })


#endif

-- method RenderNode::get_node_type
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "node"
--           , argType =
--               TInterface Name { namespace = "Gsk" , name = "RenderNode" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a render node" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface Name { namespace = "Gsk" , name = "RenderNodeType" })
-- throws : False
-- Skip return : False

foreign import ccall "gsk_render_node_get_node_type" gsk_render_node_get_node_type :: 
    Ptr RenderNode ->                       -- node : TInterface (Name {namespace = "Gsk", name = "RenderNode"})
    IO CUInt

-- | Returns the type of the render node.
renderNodeGetNodeType ::
    (B.CallStack.HasCallStack, MonadIO m, IsRenderNode a) =>
    a
    -- ^ /@node@/: a render node
    -> m Gsk.Enums.RenderNodeType
    -- ^ __Returns:__ the type of /@node@/
renderNodeGetNodeType node = liftIO $ do
    node' <- unsafeManagedPtrCastPtr node
    result <- gsk_render_node_get_node_type node'
    let result' = (toEnum . fromIntegral) result
    touchManagedPtr node
    return result'

#if defined(ENABLE_OVERLOADING)
data RenderNodeGetNodeTypeMethodInfo
instance (signature ~ (m Gsk.Enums.RenderNodeType), MonadIO m, IsRenderNode a) => O.OverloadedMethod RenderNodeGetNodeTypeMethodInfo a signature where
    overloadedMethod = renderNodeGetNodeType

instance O.OverloadedMethodInfo RenderNodeGetNodeTypeMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gsk.Objects.RenderNode.renderNodeGetNodeType",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gsk-4.0.8/docs/GI-Gsk-Objects-RenderNode.html#v:renderNodeGetNodeType"
        })


#endif

-- method RenderNode::get_opaque_rect
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType =
--               TInterface Name { namespace = "Gsk" , name = "RenderNode" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a render node" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "out_opaque"
--           , argType =
--               TInterface Name { namespace = "Graphene" , name = "Rect" }
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "return location for the opaque rect"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = True
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gsk_render_node_get_opaque_rect" gsk_render_node_get_opaque_rect :: 
    Ptr RenderNode ->                       -- self : TInterface (Name {namespace = "Gsk", name = "RenderNode"})
    Ptr Graphene.Rect.Rect ->               -- out_opaque : TInterface (Name {namespace = "Graphene", name = "Rect"})
    IO CInt

-- | Gets an opaque rectangle inside the node that GTK can determine to
-- be fully opaque.
-- 
-- There is no guarantee that this is indeed the largest opaque rectangle or
-- that regions outside the rectangle are not opaque. This function is a best
-- effort with that goal.
-- 
-- The rectangle will be fully contained in the bounds of the node.
-- 
-- /Since: 4.16/
renderNodeGetOpaqueRect ::
    (B.CallStack.HasCallStack, MonadIO m, IsRenderNode a) =>
    a
    -- ^ /@self@/: a render node
    -> m ((Bool, Graphene.Rect.Rect))
    -- ^ __Returns:__ true if part or all of the rendernode is opaque, false if no
    --   opaque region could be found.
renderNodeGetOpaqueRect self = liftIO $ do
    self' <- unsafeManagedPtrCastPtr self
    outOpaque <- SP.callocBoxedBytes 16 :: IO (Ptr Graphene.Rect.Rect)
    result <- gsk_render_node_get_opaque_rect self' outOpaque
    let result' = (/= 0) result
    outOpaque' <- (wrapBoxed Graphene.Rect.Rect) outOpaque
    touchManagedPtr self
    return (result', outOpaque')

#if defined(ENABLE_OVERLOADING)
data RenderNodeGetOpaqueRectMethodInfo
instance (signature ~ (m ((Bool, Graphene.Rect.Rect))), MonadIO m, IsRenderNode a) => O.OverloadedMethod RenderNodeGetOpaqueRectMethodInfo a signature where
    overloadedMethod = renderNodeGetOpaqueRect

instance O.OverloadedMethodInfo RenderNodeGetOpaqueRectMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gsk.Objects.RenderNode.renderNodeGetOpaqueRect",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gsk-4.0.8/docs/GI-Gsk-Objects-RenderNode.html#v:renderNodeGetOpaqueRect"
        })


#endif

-- method RenderNode::ref
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "node"
--           , argType =
--               TInterface Name { namespace = "Gsk" , name = "RenderNode" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a render node" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gsk" , name = "RenderNode" })
-- throws : False
-- Skip return : False

foreign import ccall "gsk_render_node_ref" gsk_render_node_ref :: 
    Ptr RenderNode ->                       -- node : TInterface (Name {namespace = "Gsk", name = "RenderNode"})
    IO (Ptr RenderNode)

-- | Acquires a reference on the given @GskRenderNode@.
renderNodeRef ::
    (B.CallStack.HasCallStack, MonadIO m, IsRenderNode a) =>
    a
    -- ^ /@node@/: a render node
    -> m RenderNode
    -- ^ __Returns:__ the render node with an additional reference
renderNodeRef node = liftIO $ do
    node' <- unsafeManagedPtrCastPtr node
    result <- gsk_render_node_ref node'
    checkUnexpectedReturnNULL "renderNodeRef" result
    result' <- (wrapPtr RenderNode) result
    touchManagedPtr node
    return result'

#if defined(ENABLE_OVERLOADING)
data RenderNodeRefMethodInfo
instance (signature ~ (m RenderNode), MonadIO m, IsRenderNode a) => O.OverloadedMethod RenderNodeRefMethodInfo a signature where
    overloadedMethod = renderNodeRef

instance O.OverloadedMethodInfo RenderNodeRefMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gsk.Objects.RenderNode.renderNodeRef",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gsk-4.0.8/docs/GI-Gsk-Objects-RenderNode.html#v:renderNodeRef"
        })


#endif

-- method RenderNode::serialize
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "node"
--           , argType =
--               TInterface Name { namespace = "Gsk" , name = "RenderNode" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GskRenderNode`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "GLib" , name = "Bytes" })
-- throws : False
-- Skip return : False

foreign import ccall "gsk_render_node_serialize" gsk_render_node_serialize :: 
    Ptr RenderNode ->                       -- node : TInterface (Name {namespace = "Gsk", name = "RenderNode"})
    IO (Ptr GLib.Bytes.Bytes)

-- | Serializes the /@node@/ for later deserialization via
-- 'GI.Gsk.Objects.RenderNode.renderNodeDeserialize'. No guarantees are made about the format
-- used other than that the same version of GTK will be able to deserialize
-- the result of a call to 'GI.Gsk.Objects.RenderNode.renderNodeSerialize' and
-- 'GI.Gsk.Objects.RenderNode.renderNodeDeserialize' will correctly reject files it cannot open
-- that were created with previous versions of GTK.
-- 
-- The intended use of this functions is testing, benchmarking and debugging.
-- The format is not meant as a permanent storage format.
renderNodeSerialize ::
    (B.CallStack.HasCallStack, MonadIO m, IsRenderNode a) =>
    a
    -- ^ /@node@/: a @GskRenderNode@
    -> m GLib.Bytes.Bytes
    -- ^ __Returns:__ a @GBytes@ representing the node.
renderNodeSerialize node = liftIO $ do
    node' <- unsafeManagedPtrCastPtr node
    result <- gsk_render_node_serialize node'
    checkUnexpectedReturnNULL "renderNodeSerialize" result
    result' <- (wrapBoxed GLib.Bytes.Bytes) result
    touchManagedPtr node
    return result'

#if defined(ENABLE_OVERLOADING)
data RenderNodeSerializeMethodInfo
instance (signature ~ (m GLib.Bytes.Bytes), MonadIO m, IsRenderNode a) => O.OverloadedMethod RenderNodeSerializeMethodInfo a signature where
    overloadedMethod = renderNodeSerialize

instance O.OverloadedMethodInfo RenderNodeSerializeMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gsk.Objects.RenderNode.renderNodeSerialize",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gsk-4.0.8/docs/GI-Gsk-Objects-RenderNode.html#v:renderNodeSerialize"
        })


#endif

-- method RenderNode::unref
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "node"
--           , argType =
--               TInterface Name { namespace = "Gsk" , name = "RenderNode" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a render node" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gsk_render_node_unref" gsk_render_node_unref :: 
    Ptr RenderNode ->                       -- node : TInterface (Name {namespace = "Gsk", name = "RenderNode"})
    IO ()

-- | Releases a reference on the given @GskRenderNode@.
-- 
-- If the reference was the last, the resources associated to the /@node@/ are
-- freed.
renderNodeUnref ::
    (B.CallStack.HasCallStack, MonadIO m, IsRenderNode a) =>
    a
    -- ^ /@node@/: a render node
    -> m ()
renderNodeUnref node = liftIO $ do
    node' <- B.ManagedPtr.disownManagedPtr node
    gsk_render_node_unref node'
    touchManagedPtr node
    return ()

#if defined(ENABLE_OVERLOADING)
data RenderNodeUnrefMethodInfo
instance (signature ~ (m ()), MonadIO m, IsRenderNode a) => O.OverloadedMethod RenderNodeUnrefMethodInfo a signature where
    overloadedMethod = renderNodeUnref

instance O.OverloadedMethodInfo RenderNodeUnrefMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gsk.Objects.RenderNode.renderNodeUnref",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gsk-4.0.8/docs/GI-Gsk-Objects-RenderNode.html#v:renderNodeUnref"
        })


#endif

-- method RenderNode::write_to_file
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "node"
--           , argType =
--               TInterface Name { namespace = "Gsk" , name = "RenderNode" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a render node" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "filename"
--           , argType = TBasicType TFileName
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the file to save it to"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "gsk_render_node_write_to_file" gsk_render_node_write_to_file :: 
    Ptr RenderNode ->                       -- node : TInterface (Name {namespace = "Gsk", name = "RenderNode"})
    CString ->                              -- filename : TBasicType TFileName
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | This function is equivalent to calling 'GI.Gsk.Objects.RenderNode.renderNodeSerialize'
-- followed by 'GI.GLib.Functions.fileSetContents'.
-- 
-- See those two functions for details on the arguments.
-- 
-- It is mostly intended for use inside a debugger to quickly dump a render
-- node to a file for later inspection.
renderNodeWriteToFile ::
    (B.CallStack.HasCallStack, MonadIO m, IsRenderNode a) =>
    a
    -- ^ /@node@/: a render node
    -> [Char]
    -- ^ /@filename@/: the file to save it to
    -> m ()
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
renderNodeWriteToFile node filename = liftIO $ do
    node' <- unsafeManagedPtrCastPtr node
    filename' <- stringToCString filename
    onException (do
        _ <- propagateGError $ gsk_render_node_write_to_file node' filename'
        touchManagedPtr node
        freeMem filename'
        return ()
     ) (do
        freeMem filename'
     )

#if defined(ENABLE_OVERLOADING)
data RenderNodeWriteToFileMethodInfo
instance (signature ~ ([Char] -> m ()), MonadIO m, IsRenderNode a) => O.OverloadedMethod RenderNodeWriteToFileMethodInfo a signature where
    overloadedMethod = renderNodeWriteToFile

instance O.OverloadedMethodInfo RenderNodeWriteToFileMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gsk.Objects.RenderNode.renderNodeWriteToFile",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gsk-4.0.8/docs/GI-Gsk-Objects-RenderNode.html#v:renderNodeWriteToFile"
        })


#endif

-- method RenderNode::deserialize
-- method type : MemberFunction
-- Args: [ Arg
--           { argCName = "bytes"
--           , argType = TInterface Name { namespace = "GLib" , name = "Bytes" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the bytes containing the data"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "error_func"
--           , argType =
--               TInterface Name { namespace = "Gsk" , name = "ParseErrorFunc" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "callback on parsing errors"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeCall
--           , argClosure = 2
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "user_data"
--           , argType = TBasicType TPtr
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "user_data for @error_func"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gsk" , name = "RenderNode" })
-- throws : False
-- Skip return : False

foreign import ccall "gsk_render_node_deserialize" gsk_render_node_deserialize :: 
    Ptr GLib.Bytes.Bytes ->                 -- bytes : TInterface (Name {namespace = "GLib", name = "Bytes"})
    FunPtr Gsk.Callbacks.C_ParseErrorFunc -> -- error_func : TInterface (Name {namespace = "Gsk", name = "ParseErrorFunc"})
    Ptr () ->                               -- user_data : TBasicType TPtr
    IO (Ptr RenderNode)

-- | Loads data previously created via 'GI.Gsk.Objects.RenderNode.renderNodeSerialize'.
-- 
-- For a discussion of the supported format, see that function.
renderNodeDeserialize ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    GLib.Bytes.Bytes
    -- ^ /@bytes@/: the bytes containing the data
    -> Maybe (Gsk.Callbacks.ParseErrorFunc)
    -- ^ /@errorFunc@/: callback on parsing errors
    -> m (Maybe RenderNode)
    -- ^ __Returns:__ a new render node
renderNodeDeserialize bytes errorFunc = liftIO $ do
    bytes' <- unsafeManagedPtrGetPtr bytes
    maybeErrorFunc <- case errorFunc of
        Nothing -> return FP.nullFunPtr
        Just jErrorFunc -> do
            jErrorFunc' <- Gsk.Callbacks.mk_ParseErrorFunc (Gsk.Callbacks.wrap_ParseErrorFunc Nothing (Gsk.Callbacks.drop_closures_ParseErrorFunc jErrorFunc))
            return jErrorFunc'
    let userData = nullPtr
    result <- gsk_render_node_deserialize bytes' maybeErrorFunc userData
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (wrapPtr RenderNode) result'
        return result''
    safeFreeFunPtr $ castFunPtrToPtr maybeErrorFunc
    touchManagedPtr bytes
    return maybeResult

#if defined(ENABLE_OVERLOADING)
#endif


