﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/codepipeline/CodePipeline_EXPORTS.h>
#include <aws/codepipeline/model/Tag.h>
#include <aws/codepipeline/model/WebhookDefinition.h>
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace CodePipeline {
namespace Model {

/**
 * <p>The detail returned for each webhook after listing webhooks, such as the
 * webhook URL, the webhook name, and the webhook ARN.</p><p><h3>See Also:</h3>
 * <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/codepipeline-2015-07-09/ListWebhookItem">AWS
 * API Reference</a></p>
 */
class ListWebhookItem {
 public:
  AWS_CODEPIPELINE_API ListWebhookItem() = default;
  AWS_CODEPIPELINE_API ListWebhookItem(Aws::Utils::Json::JsonView jsonValue);
  AWS_CODEPIPELINE_API ListWebhookItem& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_CODEPIPELINE_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The detail returned for each webhook, such as the webhook authentication type
   * and filter rules.</p>
   */
  inline const WebhookDefinition& GetDefinition() const { return m_definition; }
  inline bool DefinitionHasBeenSet() const { return m_definitionHasBeenSet; }
  template <typename DefinitionT = WebhookDefinition>
  void SetDefinition(DefinitionT&& value) {
    m_definitionHasBeenSet = true;
    m_definition = std::forward<DefinitionT>(value);
  }
  template <typename DefinitionT = WebhookDefinition>
  ListWebhookItem& WithDefinition(DefinitionT&& value) {
    SetDefinition(std::forward<DefinitionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A unique URL generated by CodePipeline. When a POST request is made to this
   * URL, the defined pipeline is started as long as the body of the post request
   * satisfies the defined authentication and filtering conditions. Deleting and
   * re-creating a webhook makes the old URL invalid and generates a new one.</p>
   */
  inline const Aws::String& GetUrl() const { return m_url; }
  inline bool UrlHasBeenSet() const { return m_urlHasBeenSet; }
  template <typename UrlT = Aws::String>
  void SetUrl(UrlT&& value) {
    m_urlHasBeenSet = true;
    m_url = std::forward<UrlT>(value);
  }
  template <typename UrlT = Aws::String>
  ListWebhookItem& WithUrl(UrlT&& value) {
    SetUrl(std::forward<UrlT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The text of the error message about the webhook.</p>
   */
  inline const Aws::String& GetErrorMessage() const { return m_errorMessage; }
  inline bool ErrorMessageHasBeenSet() const { return m_errorMessageHasBeenSet; }
  template <typename ErrorMessageT = Aws::String>
  void SetErrorMessage(ErrorMessageT&& value) {
    m_errorMessageHasBeenSet = true;
    m_errorMessage = std::forward<ErrorMessageT>(value);
  }
  template <typename ErrorMessageT = Aws::String>
  ListWebhookItem& WithErrorMessage(ErrorMessageT&& value) {
    SetErrorMessage(std::forward<ErrorMessageT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The number code of the error.</p>
   */
  inline const Aws::String& GetErrorCode() const { return m_errorCode; }
  inline bool ErrorCodeHasBeenSet() const { return m_errorCodeHasBeenSet; }
  template <typename ErrorCodeT = Aws::String>
  void SetErrorCode(ErrorCodeT&& value) {
    m_errorCodeHasBeenSet = true;
    m_errorCode = std::forward<ErrorCodeT>(value);
  }
  template <typename ErrorCodeT = Aws::String>
  ListWebhookItem& WithErrorCode(ErrorCodeT&& value) {
    SetErrorCode(std::forward<ErrorCodeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The date and time a webhook was last successfully triggered, in timestamp
   * format.</p>
   */
  inline const Aws::Utils::DateTime& GetLastTriggered() const { return m_lastTriggered; }
  inline bool LastTriggeredHasBeenSet() const { return m_lastTriggeredHasBeenSet; }
  template <typename LastTriggeredT = Aws::Utils::DateTime>
  void SetLastTriggered(LastTriggeredT&& value) {
    m_lastTriggeredHasBeenSet = true;
    m_lastTriggered = std::forward<LastTriggeredT>(value);
  }
  template <typename LastTriggeredT = Aws::Utils::DateTime>
  ListWebhookItem& WithLastTriggered(LastTriggeredT&& value) {
    SetLastTriggered(std::forward<LastTriggeredT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the webhook.</p>
   */
  inline const Aws::String& GetArn() const { return m_arn; }
  inline bool ArnHasBeenSet() const { return m_arnHasBeenSet; }
  template <typename ArnT = Aws::String>
  void SetArn(ArnT&& value) {
    m_arnHasBeenSet = true;
    m_arn = std::forward<ArnT>(value);
  }
  template <typename ArnT = Aws::String>
  ListWebhookItem& WithArn(ArnT&& value) {
    SetArn(std::forward<ArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies the tags applied to the webhook.</p>
   */
  inline const Aws::Vector<Tag>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Vector<Tag>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Vector<Tag>>
  ListWebhookItem& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsT = Tag>
  ListWebhookItem& AddTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace_back(std::forward<TagsT>(value));
    return *this;
  }
  ///@}
 private:
  WebhookDefinition m_definition;

  Aws::String m_url;

  Aws::String m_errorMessage;

  Aws::String m_errorCode;

  Aws::Utils::DateTime m_lastTriggered{};

  Aws::String m_arn;

  Aws::Vector<Tag> m_tags;
  bool m_definitionHasBeenSet = false;
  bool m_urlHasBeenSet = false;
  bool m_errorMessageHasBeenSet = false;
  bool m_errorCodeHasBeenSet = false;
  bool m_lastTriggeredHasBeenSet = false;
  bool m_arnHasBeenSet = false;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace CodePipeline
}  // namespace Aws
