/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/. */

use num_derive::FromPrimitive;

// Enumeration representing all crash annotations
#[allow(dead_code, non_camel_case_types, clippy::upper_case_acronyms)]
#[repr(u32)]
#[derive(Clone, Copy, Debug, FromPrimitive, PartialEq)]
pub(crate) enum CrashAnnotation {
    AbortMessage = 0,
    Accessibility = 1,
    AccessibilityClient = 2,
    AccessibilityInProcClient = 3,
    AdapterDeviceID = 4,
    AdapterDriverVendor = 5,
    AdapterDriverVersion = 6,
    AdapterSubsysID = 7,
    AdapterVendorID = 8,
    additional_minidumps = 9,
    Addons = 10,
    Android_Board = 11,
    Android_Brand = 12,
    Android_CPU_ABI = 13,
    Android_CPU_ABI2 = 14,
    Android_Device = 15,
    Android_Display = 16,
    Android_Fingerprint = 17,
    Android_Hardware = 18,
    Android_Manufacturer = 19,
    Android_Model = 20,
    Android_PackageName = 21,
    Android_ProcessName = 22,
    Android_Version = 23,
    AndroidComponentVersion = 24,
    AppInitDLLs = 25,
    ApplicationBuildID = 26,
    ApplicationServicesVersion = 27,
    AsyncShutdownTimeout = 28,
    AvailablePageFile = 29,
    AvailablePhysicalMemory = 30,
    AvailableSwapMemory = 31,
    AvailableVirtualMemory = 32,
    BackgroundTaskMode = 33,
    BackgroundTaskName = 34,
    BlockedDllList = 35,
    BlocklistInitFailed = 36,
    Breadcrumbs = 37,
    BuildID = 38,
    Comments = 39,
    ContentSandboxCapabilities = 40,
    ContentSandboxCapable = 41,
    ContentSandboxEnabled = 42,
    ContentSandboxLevel = 43,
    ContentSandboxWin32kState = 44,
    CPUMicrocodeVersion = 45,
    CrashTime = 46,
    CrashType = 47,
    CycleCollector = 48,
    DesktopEnvironment = 49,
    DeviceResetReason = 50,
    DistributionID = 51,
    DOMFissionEnabled = 52,
    DOMIPCEnabled = 53,
    DumperError = 54,
    EMCheckCompatibility = 55,
    EventLoopNestingLevel = 56,
    FontName = 57,
    GeckoViewVersion = 58,
    GleanVersion = 59,
    GMPLibraryPath = 60,
    GMPPlugin = 61,
    GPUProcessLaunchCount = 62,
    GPUProcessStatus = 63,
    GpuSandboxLevel = 64,
    GraphicsCompileShader = 65,
    GraphicsCriticalError = 66,
    GraphicsDrawShader = 67,
    GraphicsNumActiveRenderers = 68,
    GraphicsNumRenderers = 69,
    GraphicsStartupTest = 70,
    Hang = 71,
    HasDeviceTouchScreen = 72,
    HeadlessMode = 73,
    InstallTime = 74,
    ipc_channel_error = 75,
    IpcCreatePipeCloExecErrno = 76,
    IpcCreatePipeFcntlErrno = 77,
    IpcCreatePipeSocketPairErrno = 78,
    IPCFatalErrorMsg = 79,
    IPCFatalErrorProtocol = 80,
    IPCMessageLargeBufferShmemFailureSize = 81,
    IPCMessageName = 82,
    IPCMessageSize = 83,
    IPCReadErrorReason = 84,
    IPCShutdownState = 85,
    IPCSystemError = 86,
    IsGarbageCollecting = 87,
    IsWayland = 88,
    IsWebRenderResourcePathOverridden = 89,
    JavaException = 90,
    JavaStackTrace = 91,
    JSActorMessage = 92,
    JSActorName = 93,
    JSLargeAllocationFailure = 94,
    JSModuleLoadError = 95,
    JSOutOfMemory = 96,
    LastInteractionDuration = 97,
    LastStartupWasCrash = 98,
    LauncherProcessState = 99,
    LinuxUnderMemoryPressure = 100,
    LowPhysicalMemoryEvents = 101,
    MacAvailableMemorySysctl = 102,
    MacMemoryPressure = 103,
    MacMemoryPressureCriticalTime = 104,
    MacMemoryPressureNormalTime = 105,
    MacMemoryPressureSysctl = 106,
    MacMemoryPressureWarningTime = 107,
    MainThreadRunnableName = 108,
    Marionette = 109,
    MemtestOutput = 110,
    ModuleSignatureInfo = 111,
    MozCrashReason = 112,
    NimbusEnrollments = 113,
    Notes = 114,
    OOMAllocationSize = 115,
    PHCAllocStack = 116,
    PHCBaseAddress = 117,
    PHCFreeStack = 118,
    PHCKind = 119,
    PHCUsableSize = 120,
    PluginFilename = 121,
    PluginName = 122,
    PluginVersion = 123,
    ProcessType = 124,
    ProductID = 125,
    ProductName = 126,
    ProfileDirectory = 127,
    ProfilerChildShutdownPhase = 128,
    PurgeablePhysicalMemory = 129,
    QuotaManagerShutdownTimeout = 130,
    QuotaManagerStorageIsNetworkResource = 131,
    RDDProcessStatus = 132,
    ReleaseChannel = 133,
    RemoteAgent = 134,
    RemoteType = 135,
    SafeMode = 136,
    SecondsSinceLastCrash = 137,
    ServerURL = 138,
    ShutdownProgress = 139,
    ShutdownReason = 140,
    StackTraces = 141,
    StartupCacheValid = 142,
    StartupCrash = 143,
    StartupTime = 144,
    StorageConnectionNotClosed = 145,
    SubmittedFrom = 146,
    SystemMemoryUsePercentage = 147,
    TelemetryClientId = 148,
    TelemetryEnvironment = 149,
    TelemetryProfileGroupId = 150,
    TelemetryServerURL = 151,
    TelemetrySessionId = 152,
    TestBoolean = 153,
    TestInteger = 154,
    TestKey = 155,
    TestUnicode = 156,
    TextureUsage = 157,
    Throttleable = 158,
    TotalPageFile = 159,
    TotalPhysicalMemory = 160,
    TotalVirtualMemory = 161,
    UnknownNetAddrSocketFamily = 162,
    UptimeTS = 163,
    URL = 164,
    URLSegments = 165,
    User32BeforeBlocklist = 166,
    useragent_locale = 167,
    UtilityActorsName = 168,
    UtilityProcessStatus = 169,
    Vendor = 170,
    Version = 171,
    VRProcessStatus = 172,
    WasmLibrarySandboxMallocFailed = 173,
    WindowsErrorReporting = 174,
    WindowsFileDialogErrorCode = 175,
    WindowsPackageFamilyName = 176,
    Winsock_LSP = 177,
    XPCOMSpinEventLoopStack = 178,
    Count = 179,
}

impl std::fmt::Display for CrashAnnotation {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        write!(f, "{}", CRASH_ANNOTATION_STRINGS[*self as usize])
    }
}

// Type of each annotation
#[allow(dead_code)]
#[derive(Clone, Copy, PartialEq)]
pub(crate) enum CrashAnnotationType {
  String = 0,   // Any type of string, const char*, nsCString, etc...
  Boolean = 1,  // Stored as a byte
  U32 = 2,      // C/C++'s uint32_t or Rust's u32
  U64 = 3,      // C/C++'s uint64_t or Rust's u64
  USize = 4,    // C/C++'s size_t or Rust's usize
  Object = 5,   // Not usable via the Rust API
}

// Type of each annotation
static CRASH_ANNOTATION_TYPES: &[CrashAnnotationType] = &[
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::USize,
    CrashAnnotationType::USize,
    CrashAnnotationType::USize,
    CrashAnnotationType::USize,
    CrashAnnotationType::Boolean,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::Boolean,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::U32,
    CrashAnnotationType::Boolean,
    CrashAnnotationType::Boolean,
    CrashAnnotationType::U32,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::U32,
    CrashAnnotationType::String,
    CrashAnnotationType::Boolean,
    CrashAnnotationType::Boolean,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::U32,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::Boolean,
    CrashAnnotationType::U32,
    CrashAnnotationType::String,
    CrashAnnotationType::U32,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::USize,
    CrashAnnotationType::USize,
    CrashAnnotationType::Boolean,
    CrashAnnotationType::String,
    CrashAnnotationType::Boolean,
    CrashAnnotationType::Boolean,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::U32,
    CrashAnnotationType::U32,
    CrashAnnotationType::U32,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::U32,
    CrashAnnotationType::String,
    CrashAnnotationType::U32,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::U32,
    CrashAnnotationType::Boolean,
    CrashAnnotationType::Boolean,
    CrashAnnotationType::Boolean,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::U64,
    CrashAnnotationType::Boolean,
    CrashAnnotationType::U32,
    CrashAnnotationType::Boolean,
    CrashAnnotationType::U32,
    CrashAnnotationType::U32,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::U32,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::Boolean,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::USize,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::U32,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::USize,
    CrashAnnotationType::String,
    CrashAnnotationType::Boolean,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::Boolean,
    CrashAnnotationType::String,
    CrashAnnotationType::Boolean,
    CrashAnnotationType::U64,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::Object,
    CrashAnnotationType::Boolean,
    CrashAnnotationType::Boolean,
    CrashAnnotationType::U64,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::U32,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::Boolean,
    CrashAnnotationType::U32,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::USize,
    CrashAnnotationType::Boolean,
    CrashAnnotationType::USize,
    CrashAnnotationType::USize,
    CrashAnnotationType::USize,
    CrashAnnotationType::U32,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::Boolean,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::Boolean,
    CrashAnnotationType::Boolean,
    CrashAnnotationType::U32,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
    CrashAnnotationType::String,
];

// Stringified representation of each annotation. Most of these will just match
// the corresponding enum values, but for historical reasons some of them are
// different in string form when stored in the .extra file.
static CRASH_ANNOTATION_STRINGS: &[&str] = &[
    "AbortMessage",
    "Accessibility",
    "AccessibilityClient",
    "AccessibilityInProcClient",
    "AdapterDeviceID",
    "AdapterDriverVendor",
    "AdapterDriverVersion",
    "AdapterSubsysID",
    "AdapterVendorID",
    "additional_minidumps",
    "Add-ons",
    "Android_Board",
    "Android_Brand",
    "Android_CPU_ABI",
    "Android_CPU_ABI2",
    "Android_Device",
    "Android_Display",
    "Android_Fingerprint",
    "Android_Hardware",
    "Android_Manufacturer",
    "Android_Model",
    "Android_PackageName",
    "Android_ProcessName",
    "Android_Version",
    "AndroidComponentVersion",
    "AppInitDLLs",
    "ApplicationBuildID",
    "ApplicationServicesVersion",
    "AsyncShutdownTimeout",
    "AvailablePageFile",
    "AvailablePhysicalMemory",
    "AvailableSwapMemory",
    "AvailableVirtualMemory",
    "BackgroundTaskMode",
    "BackgroundTaskName",
    "BlockedDllList",
    "BlocklistInitFailed",
    "Breadcrumbs",
    "BuildID",
    "Comments",
    "ContentSandboxCapabilities",
    "ContentSandboxCapable",
    "ContentSandboxEnabled",
    "ContentSandboxLevel",
    "ContentSandboxWin32kState",
    "CPUMicrocodeVersion",
    "CrashTime",
    "CrashType",
    "CycleCollector",
    "DesktopEnvironment",
    "DeviceResetReason",
    "DistributionID",
    "DOMFissionEnabled",
    "DOMIPCEnabled",
    "DumperError",
    "EMCheckCompatibility",
    "EventLoopNestingLevel",
    "FontName",
    "GeckoViewVersion",
    "GleanVersion",
    "GMPLibraryPath",
    "GMPPlugin",
    "GPUProcessLaunchCount",
    "GPUProcessStatus",
    "GpuSandboxLevel",
    "GraphicsCompileShader",
    "GraphicsCriticalError",
    "GraphicsDrawShader",
    "GraphicsNumActiveRenderers",
    "GraphicsNumRenderers",
    "GraphicsStartupTest",
    "Hang",
    "HasDeviceTouchScreen",
    "HeadlessMode",
    "InstallTime",
    "ipc_channel_error",
    "IpcCreatePipeCloExecErrno",
    "IpcCreatePipeFcntlErrno",
    "IpcCreatePipeSocketPairErrno",
    "IPCFatalErrorMsg",
    "IPCFatalErrorProtocol",
    "IPCMessageLargeBufferShmemFailureSize",
    "IPCMessageName",
    "IPCMessageSize",
    "IPCReadErrorReason",
    "IPCShutdownState",
    "IPCSystemError",
    "IsGarbageCollecting",
    "IsWayland",
    "IsWebRenderResourcePathOverridden",
    "JavaException",
    "JavaStackTrace",
    "JSActorMessage",
    "JSActorName",
    "JSLargeAllocationFailure",
    "JSModuleLoadError",
    "JSOutOfMemory",
    "LastInteractionDuration",
    "LastStartupWasCrash",
    "LauncherProcessState",
    "LinuxUnderMemoryPressure",
    "LowPhysicalMemoryEvents",
    "MacAvailableMemorySysctl",
    "MacMemoryPressure",
    "MacMemoryPressureCriticalTime",
    "MacMemoryPressureNormalTime",
    "MacMemoryPressureSysctl",
    "MacMemoryPressureWarningTime",
    "MainThreadRunnableName",
    "Marionette",
    "MemtestOutput",
    "ModuleSignatureInfo",
    "MozCrashReason",
    "NimbusEnrollments",
    "Notes",
    "OOMAllocationSize",
    "PHCAllocStack",
    "PHCBaseAddress",
    "PHCFreeStack",
    "PHCKind",
    "PHCUsableSize",
    "PluginFilename",
    "PluginName",
    "PluginVersion",
    "ProcessType",
    "ProductID",
    "ProductName",
    "ProfileDirectory",
    "ProfilerChildShutdownPhase",
    "PurgeablePhysicalMemory",
    "QuotaManagerShutdownTimeout",
    "QuotaManagerStorageIsNetworkResource",
    "RDDProcessStatus",
    "ReleaseChannel",
    "RemoteAgent",
    "RemoteType",
    "SafeMode",
    "SecondsSinceLastCrash",
    "ServerURL",
    "ShutdownProgress",
    "ShutdownReason",
    "StackTraces",
    "StartupCacheValid",
    "StartupCrash",
    "StartupTime",
    "StorageConnectionNotClosed",
    "SubmittedFrom",
    "SystemMemoryUsePercentage",
    "TelemetryClientId",
    "TelemetryEnvironment",
    "TelemetryProfileGroupId",
    "TelemetryServerURL",
    "TelemetrySessionId",
    "TestBoolean",
    "TestInteger",
    "TestKey",
    "TestUnicode",
    "TextureUsage",
    "Throttleable",
    "TotalPageFile",
    "TotalPhysicalMemory",
    "TotalVirtualMemory",
    "UnknownNetAddrSocketFamily",
    "UptimeTS",
    "URL",
    "URLSegments",
    "User32BeforeBlocklist",
    "useragent_locale",
    "UtilityActorsName",
    "UtilityProcessStatus",
    "Vendor",
    "Version",
    "VRProcessStatus",
    "WasmLibrarySandboxMallocFailed",
    "WindowsErrorReporting",
    "WindowsFileDialogErrorCode",
    "WindowsPackageFamilyName",
    "Winsock_LSP",
    "XPCOMSpinEventLoopStack",
];

// Annotations which should be skipped when they have specific values
struct CrashAnnotationSkipValue {
  annotation: CrashAnnotation,
  value: &'static [u8],
}

static CRASH_ANNOTATIONS_SKIP_VALUES: &[CrashAnnotationSkipValue] = &[
    CrashAnnotationSkipValue { annotation: CrashAnnotation::BlocklistInitFailed, value: b"0" },
    CrashAnnotationSkipValue { annotation: CrashAnnotation::EventLoopNestingLevel, value: b"0" },
    CrashAnnotationSkipValue { annotation: CrashAnnotation::IsGarbageCollecting, value: b"0" },
    CrashAnnotationSkipValue { annotation: CrashAnnotation::LowPhysicalMemoryEvents, value: b"0" },
    CrashAnnotationSkipValue { annotation: CrashAnnotation::OOMAllocationSize, value: b"0" },
    CrashAnnotationSkipValue { annotation: CrashAnnotation::TextureUsage, value: b"0" },
    CrashAnnotationSkipValue { annotation: CrashAnnotation::User32BeforeBlocklist, value: b"0" },
];


/// Returns the type of a crash annotation.
///
/// # Arguments
///
/// * `annotation` - a crash annotation
pub(crate) fn type_of_annotation(annotation: CrashAnnotation) -> CrashAnnotationType {
  CRASH_ANNOTATION_TYPES[annotation as usize]
}

/// Checks if the annotation should be included. Some annotations are skipped
/// if their value matches a specific one (like the value 0).
///
/// # Arguments
///
/// * `annotation` - the crash annotation to be checked
/// * `value` - the contents of the annotation as a string
pub(crate) fn should_include_annotation(annotation: CrashAnnotation, value: &[u8]) -> bool {
    if let Some(skip_value) = CRASH_ANNOTATIONS_SKIP_VALUES
        .iter()
        .find(|&a| a.annotation == annotation)
    {
        skip_value.value != value
    } else {
        true
    }
}
